# Delete Confirmation Modal Usage Guide

This reusable delete confirmation modal can be used across different modules in the application.

## Basic Usage

### 1. Include the Modal Component

Add this to your view file:

```blade
@include('partials.deleteConfirmationModal')
```

### 2. Include the JavaScript

Add this to your scripts section:

```blade
<script src="{{ asset('js/deleteConfirmationModal.js') }}"></script>
```

### 3. Use the Modal in Your Buttons

Replace your existing delete buttons with:

```blade
<button type="button" class="btn btn-danger"
    data-delete-url="{{ route('your.delete.route', $item->id) }}"
    data-delete-message="Custom message for this specific delete action">
    <i class="fa fa-trash"></i> Delete
</button>
```

## Advanced Usage

### Custom Messages

You can provide custom messages for different delete actions:

```blade
<!-- For single record deletion -->
<button type="button" class="btn btn-danger"
    data-delete-url="{{ route('admin.products.destroy', $product->id) }}"
    data-delete-message="Are you sure you want to delete this product? This will also delete all related stock records.">
    <i class="fa fa-trash"></i>
</button>

<!-- For bulk deletion -->
<button type="button" class="btn btn-danger"
    data-delete-url="{{ route('admin.products.massDestroy') }}"
    data-delete-message="Are you sure you want to delete all selected products? This action cannot be undone.">
    <i class="fa fa-trash"></i> Delete Selected
</button>
```

### JavaScript Integration

You can also trigger the modal programmatically:

```javascript
// Show delete confirmation
showDeleteConfirmation(
    "/admin/products/1",
    "Are you sure you want to delete this product?"
);

// For bulk operations
showDeleteConfirmation(
    "/admin/products/massDestroy",
    "Are you sure you want to delete all selected products?"
);
```

## Controller Requirements

Your controller methods should handle AJAX requests and return JSON responses:

```php
public function destroy(YourModel $item)
{
    try {
        $item->delete();

        if (request()->ajax()) {
            return response()->json([
                'status' => 'success',
                'message' => 'Record deleted successfully'
            ]);
        }

        return redirect()->back()->with('message', 'Record deleted successfully');
    } catch (\Exception $e) {
        if (request()->ajax()) {
            return response()->json([
                'status' => 'error',
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }

        return redirect()->back()->with('error', 'An error occurred');
    }
}
```

## Features

-   ✅ Reusable across all modules
-   ✅ Customizable messages
-   ✅ AJAX-based deletion
-   ✅ Loading states
-   ✅ Error handling
-   ✅ Success notifications
-   ✅ Bootstrap modal styling
-   ✅ Font Awesome icons
-   ✅ Toastr notifications support

## Styling

The modal uses Bootstrap 4 classes and can be customized by modifying the `deleteConfirmationModal.blade.php` file.

## Dependencies

-   jQuery
-   Bootstrap 4
-   Font Awesome (optional)
-   Toastr (optional, for notifications)
