<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Cache;

class SettingsService
{
    /**
     * Get all settings with caching
     */
    public static function getAll()
    {
        return Cache::remember('settings', 3600, function () {
            return Setting::getSettings();
        });
    }

    /**
     * Get a specific setting value
     */
    public static function get($key, $default = null)
    {
        $settings = static::getAll();
        return $settings->$key ?? $default;
    }

    /**
     * Get app name
     */
    public static function getAppName()
    {
        return static::get('app_name', 'BlueWeld SG');
    }

    /**
     * Get logo URL
     */
    public static function getLogoUrl()
    {
        $logo = static::get('logo');
        if ($logo) {
            return asset('storage/' . $logo);
        }

        // Return default Alphinex logo if no custom logo is set
        $defaultLogo = 'logos/alphinex-logo.png';
        if (file_exists(public_path('storage/' . $defaultLogo))) {
            return asset('storage/' . $defaultLogo);
        }

        return null;
    }

    /**
     * Get address
     */
    public static function getAddress()
    {
        return static::get('address');
    }

    /**
     * Get address
     */
    public static function getSymbol()
    {
        return static::get('currency_symbol', '$');
    }

    /**
     * Clear settings cache
     */
    public static function clearCache()
    {
        Cache::forget('settings');
    }

    /**
     * Update settings and clear cache
     */
    public static function update($data)
    {
        $settings = Setting::getSettings();
        $settings->update($data);
        static::clearCache();
        return $settings;
    }
}
